<?php

namespace xo\content;

use Shanept\MimeReader;
use xo\cms\DataList;
use xo\cms\DatalistItem;
use xo\cms\Path;
use xo\cms\Resource;
use xo\cms\ResourceData;
use xo\cms\ResourceRef;
use xo\cms\WebPage;
use xo\cms\WebPageModel;
use xo\cms\WebSite;

class ContentService {
	private $config;
	public function __construct($config) {
		$this->config = $config;
	}
	public static function uuid() {
		return hex2bin ( sprintf ( '%04x%04x%04x%04x%04x%04x%04x%04x', mt_rand ( 0, 0xffff ), mt_rand ( 0, 0xffff ), mt_rand ( 0, 0xffff ), mt_rand ( 0, 0x0fff ) | 0x4000, mt_rand ( 0, 0x3fff ) | 0x8000, mt_rand ( 0, 0xffff ), mt_rand ( 0, 0xffff ), mt_rand ( 0, 0xffff ) ) );
	}
	private function getDataFilePath($id) {
		$id = bin2hex ( $id );
		return $this->config->getDataDir () . '/' . substr ( $id, 0, 2 ) . '/' . substr ( $id, 2, 3 ) . '/' . $id;
	}
	private function createResourceRow($db, Resource $resource, $i18nCode = null) {
		if ($i18nCode == null) {
			$i18nCode = self::uuid ();
		}
		$query = 'INSERT INTO CONTENT_RESOURCE_V2(id,i18n_code,locale,creation_date,path_id) values(:id,:i18nCode,:locale,:version,:path_id)';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $resource->id );
		$stmt->bindValue ( ":i18nCode", $i18nCode );
		$stmt->bindValue ( ":locale", $resource->locale );
		$stmt->bindValue ( ":version", $resource->version );
		$stmt->bindValue ( ":path_id", $resource->path->id );
		$stmt->execute ();
	}
	private function createResourceRefRow($db, ResourceRef $ref) {
		$query = 'INSERT INTO CONTENT_RESOURCE_REF_V2(id,comment,label,description,title,img_id,img_over_id,img_alt,path_id,url) 
				values(:id,:comment,:label,:description,:title,:img_id,:img_over_id,:img_alt,:path_id,:url)';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $ref->id );
		$stmt->bindValue ( ":comment", $ref->comment );
		$stmt->bindValue ( ":label", $ref->label );
		$stmt->bindValue ( ":description", $ref->description );
		$stmt->bindValue ( ":title", $ref->title );
		$stmt->bindValue ( ":img_id", $ref->img == null ? null : $ref->img->id );
		$stmt->bindValue ( ":img_over_id", $ref->imgOver == null ? null : $ref->imgOver->id );
		$stmt->bindValue ( ":img_alt", $ref->imgAlt );
		$stmt->bindValue ( ":path_id", $ref->path == null ? null : $ref->path->id );
		$stmt->bindValue ( ":url", $ref->resource == null ? $ref->url : null );
		$stmt->execute ();
	}
	private function updateResourceRefRow($db, ResourceRef $ref) {
	    $query = 'UPDATE CONTENT_RESOURCE_REF_V2 set comment=:comment,label=:label,description=:description,title=:title,img_id=:img_id,img_over_id=:img_over_id,img_alt=:img_alt,path_id=:path_id,url=:url WHERE id=:id';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $ref->id );
		$stmt->bindValue ( ":comment", $ref->comment );
		$stmt->bindValue ( ":label", $ref->label );
		$stmt->bindValue ( ":description", $ref->description );
		$stmt->bindValue ( ":title", $ref->title );
		$stmt->bindValue ( ":img_id", $ref->img == null ? null : $ref->img->id );
		$stmt->bindValue ( ":img_over_id", $ref->imgOver == null ? null : $ref->imgOver->id );
		$stmt->bindValue ( ":img_alt", $ref->imgAlt );
		$stmt->bindValue ( ":path_id", $ref->path == null ? null : $ref->path->id );
		$stmt->bindValue ( ":url", $ref->path == null ? $ref->url : null );
		$stmt->execute ();
	}
	private function createResourceVersionRow($db, Resource $resource) {
		$query = 'UPDATE CONTENT_RESOURCE_VERSION_V2 set current_version=:currentVersion where resource_id=:resourceId';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":currentVersion", 0 );
		$stmt->bindValue ( ":resourceId", $resource->id );
		$stmt->execute ();
		$query = 'INSERT INTO CONTENT_RESOURCE_VERSION_V2(resource_id,version,data_id,data_preview_id,name,comment,description,current_version) values(:resourceId,:version,:dataId,:dataPreviewId,:name,:comment,:description,:currentVersion)';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":resourceId", $resource->id );
		$stmt->bindValue ( ":version", $resource->version );
		$stmt->bindValue ( ":dataId", $resource->data->id );
		$stmt->bindValue ( ":dataPreviewId", $resource->preview->id );
		$stmt->bindValue ( ":name", $resource->name );
		$stmt->bindValue ( ":description", $resource->description );
		$stmt->bindValue ( ":comment", $resource->comment );
		$stmt->bindValue ( ":currentVersion", 1 );
		$stmt->execute ();
	}
	private function createWebPageRow($db, WebPage $webPage) {
		$query = 'INSERT INTO CONTENT_WEBPAGE_V2(id,template,data_list_id,parameters) values(:id,:template,:dataListId,:parameters)';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $webPage->id );
		$stmt->bindValue ( ":template", $webPage->templateName );
		$stmt->bindValue ( ":parameters", $webPage->parameters );
		$stmt->bindValue ( ":dataListId", $webPage->model->id );
		$stmt->execute ();
	}
	public function getResourceData($resourceData) {
		$content = file_get_contents ( $this->getDataFilePath ( $resourceData->id ) );
		return $content;
	}
	public function writeResourceData($out, $resourceData) {
		fwrite ( $out, $this->getResourceData ( $resourceData ) );
	}
	public function getWebPagesCount() {
		$query = 'SELECT count(*) as c FROM CONTENT_RESOURCE_V2 r left join CONTENT_WEBPAGE_V2 p on r.id = p.id  where p.template is not null';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->execute ();
		$row = $stmt->fetch ();
		return intval ( $row ["c"] );
	}
	public function getWebPages($offset = 0, $limit = 25) {
		$query = 'SELECT res.path_id,vers.*, map.path,map.online,p.template FROM CONTENT_RESOURCE_V2 res join CONTENT_RESOURCE_VERSION_V2 vers on res.id=vers.resource_id  join CONTENT_RESOURCE_MAP_V2 map on res.path_id=map.id left join CONTENT_WEBPAGE_V2 p on vers.resource_id = p.id  where current_version=:currentVersion and p.template is not null order by name LIMIT :offset,:limit';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":currentVersion", 1 );
		$stmt->bindValue ( ":offset", $offset );
		$stmt->bindValue ( ":limit", $limit );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$resources = array ();
		foreach ( $rows as $row ) {
			$resource = new WebPage ( $row ["resource_id"] );
			$resource->name = $row ["name"];
			$resource->templateName = $row ["template"];
			$resource->path = new Path ( $row ["path_id"] );
			$resource->path->online = $row ["online"];
			$resource->path->path = $row ["path"];
			$resource->data = new ResourceData ( $row ["data_id"] );
			$resource->preview = new ResourceData ( $row ['data_preview_id'] );
			$resource->version = $row ["version"];
			array_push ( $resources, $resource );
		}
		return $resources;
	}
	public function getResourcesCount($contentType = null) {
		$query = 'SELECT count(*) as c FROM CONTENT_RESOURCE_V2 r left join CONTENT_WEBPAGE_V2 p on r.id = p.id  where p.template is null';
		if ($contentType !== null) {
			$query .= 'and content_type';
		}
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		if ($contentType !== null) {
			$query .= '';
		}
		$stmt->execute ();
		$row = $stmt->fetch ();
		return intval ( $row ["c"] );
	}
	public function getContentTypes() {
		$query = 'SELECT distinct(content_type) as label,count(*) as count FROM CONTENT_RESOURCE_V2 res 
				join CONTENT_RESOURCE_VERSION_V2 vers on res.id=vers.resource_id 
				join CONTENT_RESOURCE_MAP_V2 map on res.path_id=map.id 
				left join CONTENT_WEBPAGE_V2 p on vers.resource_id = p.id 
				join CONTENT_DATA_V2 d on d.id = vers.data_id where current_version=:currentVersion and p.template is null group by content_type';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":currentVersion", 1 );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$contentTypes = array ();
		foreach ( $rows as $row ) {
			$contentType = array ();
			$contentType ["count"] = $row ['count'];
			$contentType ["label"] = $row ['label'];
			array_push ( $contentTypes, $contentType );
		}
		return $contentTypes;
	}
	public function readResources($rows, &$resources) {
		foreach ( $rows as $row ) {
			$resource = new Resource ( $row ["resource_id"] );
			$resource->name = $row ["name"];
			$resource->path = new Path ( $row ["path_id"] );
			$resource->path->online = $row ['online'] != null && $row ["online"] > 0;
			$resource->path->path = $row ["path"];
			$resource->data = new ResourceData ( $row ["data_id"] );
			$resource->preview = new ResourceData ( $row ['data_preview_id'] );
			$resource->version = $row ["version"];
			array_push ( $resources, $resource );
		}
	}
	public function filterResourcesByContentType($mimeType, $offset = 0, $limit = 25) {
		$query = 'SELECT res.path_id,vers.*, map.path,map.online FROM CONTENT_RESOURCE_V2 res 
				join CONTENT_RESOURCE_VERSION_V2 vers on res.id=vers.resource_id 
				join CONTENT_RESOURCE_MAP_V2 map on res.path_id=map.id left 
				join CONTENT_WEBPAGE_V2 p on vers.resource_id = p.id 
				join CONTENT_DATA_V2 d on d.id = vers.data_id 
				where current_version=:currentVersion
					and p.template is null 
					and d.content_type=:contentType 
				order by name LIMIT :offset,:limit';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":currentVersion", 1 );
		$stmt->bindValue ( ":offset", $offset );
		$stmt->bindValue ( ":limit", $limit );
		$stmt->bindValue ( ":contentType", $mimeType );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$resources = array ();
		$this->readResources ( $rows, $resources );
		return $resources;
	}
	public function getResources($offset = 0, $limit = 25) {
		$query = 'SELECT res.path_id,vers.*, map.path,map.online FROM CONTENT_RESOURCE_V2 res 
				join CONTENT_RESOURCE_VERSION_V2 vers on res.id=vers.resource_id  
				join CONTENT_RESOURCE_MAP_V2 map on res.path_id=map.id 
				left join CONTENT_WEBPAGE_V2 p on vers.resource_id = p.id  
				where current_version=:currentVersion and p.template is null order by name LIMIT :offset,:limit';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":currentVersion", 1 );
		$stmt->bindValue ( ":offset", $offset );
		$stmt->bindValue ( ":limit", $limit );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$resources = array ();
		$this->readResources ( $rows, $resources );
		return $resources;
	}
	public function getResource($id, $version = null) {
		if ($version === null) {
			$query = 'SELECT res.*,vers.*, map.path,map.online,p.template,p.seo_title,p.seo_description,p.seo_img_id,p.data_list_id,p.parameters FROM CONTENT_RESOURCE_V2 res join CONTENT_RESOURCE_VERSION_V2 vers on res.id=vers.resource_id  join CONTENT_RESOURCE_MAP_V2 map on res.path_id=map.id left join CONTENT_WEBPAGE_V2 p on vers.resource_id = p.id  where res.id=:id and current_version=:currentVersion';
		} else {
			// TODO
		}
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		if ($version === null) {
			$stmt->bindValue ( ":currentVersion", 1 );
		} else {
			$stmt->bindValue ( ":version", $version );
		}
		$stmt->bindValue ( ":id", $id );
		$result = $stmt->execute ();
		$row = $stmt->fetch ();
		return $this->readResource ( $row );
	}
	public function readResource($row) {
		$resource = null;
		if ($row) {
			if (array_key_exists ( "template", $row ) && ! empty ( $row ["template"] )) {
				$resource = new WebPage ( $row ["id"] );
				$resource->templateName = $row ["template"];
				$resource->seoTitle = $row ["seo_title"];
				$resource->seoDescription = $row ["seo_description"];
				$resource->seoImage = $row ["seo_img_id"] == null ? null : $this->getResource ( $row ["seo_img_id"] );
				// FIXME website listid
				$resource->model = new WebPageModel ( $this, hex2bin ( "e2cf526e46f845b0a5e7cb8d66812238" ), $row ["data_list_id"] );
				if ($row ["parameters"] !== null) {
					$resource->parameters = $row ["parameters"];
				}
			} else {
				$resource = new Resource ( $row ["id"] );
			}
			$resource->locale = $row ["locale"];
			$resource->description = $row ["description"];
			$resource->comment = $row ["comment"];
			$resource->name = $row ["name"];
			$resource->path = new Path ( $row ["path_id"] );
			$resource->path->path = $row ["path"];
			$resource->path->online = $row ["online"];
			$resource->data = new ResourceData ( $row ["data_id"] );
			$resource->preview = new ResourceData ( $row ["data_preview_id"] );
			$resource->version = $row ["version"];
		}
		
		return $resource;
	}
	public function getResourceFromPath($path) {
		$query = "select * from CONTENT_RESOURCE_MAP_V2 where path=:path";
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":path", $path );
		$result = $stmt->execute ();
		$row = $stmt->fetch ();
		return $this->getResource ( $row ["resource_id"] );
	}
	public function getCurrentResource($resourceId) {
		return $this->getResource ( $resourceId );
	}
	public function flatAncestors(ResourceAncestor $ancestor) {
		$ancestors = array ();
		$stack = array ();
		$a = $ancestor;
		while ( $a != null ) {
			if (count ( $a->ancestors ) > 0) {
				foreach ( $a->ancestors as $tmp ) {
					array_push ( $stack, $tmp );
				}
			} else {
				array_push ( $ancestors, $a );
			}
			$a = array_pop ( $stack );
		}
		
		return $ancestors;
	}
	function fetchWebPageFromDataListId($dataListId) {
		$query = "select id from CONTENT_WEBPAGE_V2
						where data_list_id=:dataListId";
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":dataListId", $dataListId );
		$stmt->execute ();
		return $stmt->fetch ();
	}
	function fetchWebSiteFromDataListId($dataListId) {
		$query = "select id from CONTENT_WEBSITE_V2
						where data_list_id=:dataListId";
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":dataListId", $dataListId );
		$stmt->execute ();
		return $stmt->fetch ();
	}
	function fetchAncestorsRows($itemValueId) {
		$query = "select * from CONTENT_LIST_ITEM_V2 item 
						where value_resource_id=:value_resource_id
						   or value_resource_ref_id=:value_resource_ref_id
						   or value_list_id=:value_list_id";
		
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":value_resource_id", $itemValueId );
		$stmt->bindValue ( ":value_resource_ref_id", $itemValueId );
		$stmt->bindValue ( ":value_list_id", $itemValueId );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		return $rows;
	}
	public function getAncestors($itemValueId) {
		$stack = array ();
		$done = array ();
		$id = $itemValueId;
		$parents = array ();
		$values = array ();
		while ( $id != null ) {
			$rows = $this->fetchAncestorsRows ( $id );
			array_push ( $done, $id );
			error_log ( count ( $rows ) );
			if ($rows == null || count ( $rows ) == 0) {
				if ($itemValueId == $id) {
					return array ();
				} else {
					$value = new DataList ( $this, $id );
					$values [$id] = $value;
					$row = $this->fetchWebPageFromDataListId ( $id );
					if (! empty ( $row )) {
						if ($row ["id"] != null) {
							$parentId = $row ["id"];
							$values [$parentId] = new WebPage ( $parentId );
						}
						
						if (! array_key_exists ( $id, $parents )) {
							$parents [$id] = array ();
						}
						if (! (in_array ( $parentId, $parents [$id] ))) {
							array_push ( $parents [$id], $parentId );
						}
					} else {
						$row = $this->fetchWebSiteFromDataListId ( $id );
						if (! empty ( $row )) {
							if ($row ["id"] != null) {
								$parentId = $row ["id"];
								$values [$parentId] = new WebSite ( $parentId );
							}
							
							if (! array_key_exists ( $id, $parents )) {
								$parents [$id] = array ();
							}
							if (! (in_array ( $parentId, $parents [$id] ))) {
								array_push ( $parents [$id], $parentId );
							}
						}
					}
				}
			} else {
				foreach ( $rows as $row ) {
					$listId = $row ["list_id"];
					error_log ( bin2hex ( $listId ) );
					$value = null;
					if ($row ["value_resource_id"] == $id) {
						$value = new Resource ( $id );
					} else if ($row ["value_resource_ref_id"] == $id) {
						$value = new ResourceRef ( $id );
					} else if ($row ["value_list_id"] == $id) {
						$value = new DataList ( $this, $id );
					}
					$values [$id] = $value;
					if (! array_key_exists ( $id, $parents )) {
						$parents [$id] = array ();
					}
					if (! (in_array ( $listId, $parents [$id] ))) {
						array_push ( $parents [$id], $listId );
					}
					
					if (! (in_array ( $listId, $done ) || in_array ( $listId, $stack ))) {
						array_push ( $stack, $listId );
					}
				}
			}
			$id = array_pop ( $stack );
		}
		
		// Flatten :
		$stack = array ();
		$path = array ();
		$node = $values [$itemValueId];
		$ancestors = array ();
		while ( $node != null ) {
			array_push ( $path, $node );
			$p = "";
			foreach ( $path as $pathElt ) {
				$p .= ">";
				$p .= $pathElt->id;
			}
			error_log ( $p );
			if (array_key_exists ( $node->id, $parents )) {
				foreach ( $parents [$node->id] as $parentId ) {
					array_push ( $stack, $values [$parentId] );
				}
			} else {
				// add path to ancestors
				$ancestor = array ();
				foreach ( $path as $pathElt ) {
					array_push ( $ancestor, $pathElt );
				}
				array_push ( $ancestors, array_reverse ( $ancestor ) );
				array_pop ( $path );
			}
			$node = array_pop ( $stack );
		}
		return $ancestors;
	}
	public function getAncestorsOK($itemValueId) {
		$stack = array ();
		$done = array ();
		$id = $itemValueId;
		$parents = array ();
		$values = array ();
		while ( $id != null ) {
			$rows = $this->fetchAncestorsRows ( $id );
			array_push ( $done, $id );
			error_log ( count ( $rows ) );
			if ($rows == null || count ( $rows ) == 0) {
				if ($itemValueId == $id) {
					return array ();
				} else {
					$value = new DataList ( $this, $id );
					$values [$id] = $value;
					$row = $this->fetchWebPageFromDataListId ( $id );
					if (! empty ( $row )) {
						if ($row ["id"] != null) {
							$parentId = $row ["id"];
							$values [$parentId] = new WebPage ( $parentId );
						}
						
						if (! array_key_exists ( $id, $parents )) {
							$parents [$id] = array ();
						}
						if (! (in_array ( $parentId, $parents [$id] ))) {
							array_push ( $parents [$id], $parentId );
						}
					} else {
						$row = $this->fetchWebSiteFromDataListId ( $id );
						if (! empty ( $row )) {
							if ($row ["id"] != null) {
								$parentId = $row ["id"];
								$values [$parentId] = new WebSite ( $parentId );
							}
							
							if (! array_key_exists ( $id, $parents )) {
								$parents [$id] = array ();
							}
							if (! (in_array ( $parentId, $parents [$id] ))) {
								array_push ( $parents [$id], $parentId );
							}
						}
					}
				}
			} else {
				foreach ( $rows as $row ) {
					$listId = $row ["list_id"];
					error_log ( bin2hex ( $listId ) );
					$value = null;
					if ($row ["value_resource_id"] == $id) {
						$value = new Resource ( $id );
					} else if ($row ["value_resource_ref_id"] == $id) {
						$value = new ResourceRef ( $id );
					} else if ($row ["value_list_id"] == $id) {
						$value = new DataList ( $this, $id );
					}
					$values [$id] = $value;
					if (! array_key_exists ( $id, $parents )) {
						$parents [$id] = array ();
					}
					if (! (in_array ( $listId, $parents [$id] ))) {
						array_push ( $parents [$id], $listId );
					}
					
					if (! (in_array ( $listId, $done ) || in_array ( $listId, $stack ))) {
						array_push ( $stack, $listId );
					}
				}
			}
			$id = array_pop ( $stack );
		}
		
		// Flatten :
		$stack = array ();
		$path = array ();
		$node = $values [$itemValueId];
		$ancestors = array ();
		while ( $node != null ) {
			array_push ( $path, $node );
			$p = "";
			foreach ( $path as $pathElt ) {
				$p .= ">";
				$p .= $pathElt->id;
			}
			error_log ( $p );
			if (array_key_exists ( $node->id, $parents )) {
				foreach ( $parents [$node->id] as $parentId ) {
					array_push ( $stack, $values [$parentId] );
				}
			} else {
				// add path to ancestors
				$ancestor = array ();
				foreach ( $path as $pathElt ) {
					array_push ( $ancestor, $pathElt );
				}
				array_push ( $ancestors, array_reverse ( $ancestor ) );
				array_pop ( $path );
			}
			$node = array_pop ( $stack );
		}
		return $ancestors;
	}
	private function createResourceMapRow($db, Path $path) {
		$query = "INSERT INTO CONTENT_RESOURCE_MAP_V2(id,resource_id,path,online) value(:id,:resourceId,:path,:online)";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $path->id );
		$stmt->bindValue ( ":resourceId", $path->resource->id );
		$stmt->bindValue ( ":path", $path->path );
		$stmt->bindValue ( ":online", $path->online ? 1 : 0 );
		$stmt->execute ();
	}
	private function createDataRow($db, ResourceData $data) {
		$query = "INSERT INTO CONTENT_DATA_V2(id,content_type,filename,date,charset,encoding,md5,length) 
				value(:id,:content_type,:filename,:date,:charset,:encoding,:md5,:length)";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $data->id );
		$stmt->bindValue ( ":content_type", $data->contentType );
		$stmt->bindValue ( ":filename", $data->filename );
		$stmt->bindValue ( ":date", $data->date );
		$stmt->bindValue ( ":charset", $data->charset );
		$stmt->bindValue ( ":encoding", $data->encoding );
		$stmt->bindValue ( ":md5", $data->md5 );
		$stmt->bindValue ( ":length", $data->length );
		$stmt->execute ();
	}
	public function updateResourceMap($pathId, $path, $online = 0) {
		$query = "UPDATE CONTENT_RESOURCE_MAP_V2 SET path=:path, online=:online where id=:pathId";
		try {
			$db = $this->config->db ();
			$stmt = $db->prepare ( $query );
			$stmt->bindValue ( ":pathId", $pathId );
			$stmt->bindValue ( ":path", $path );
			$stmt->bindValue ( ":online", $online );
			$stmt->execute ();
		} catch ( Exception $e ) {
			throw $e;
		}
	}
	public function getResourceRefsCount() {
		$query = 'SELECT count(*) as c FROM CONTENT_RESOURCE_REF_V2';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->execute ();
		$row = $stmt->fetch ();
		return intval ( $row ["c"] );
	}
	public function getResourceRefsWithImgOver() {
		$query = 'SELECT ref.img_id,ref.id,ref.img_over_id,m.path ,m.id as path_id
FROM CONTENT_RESOURCE_REF_V2 ref 
join CONTENT_RESOURCE_V2 res on ref.img_over_id=res.id 
join CONTENT_RESOURCE_MAP_V2 m on res.path_id=m.id';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$refs = array ();
		foreach ( $rows as $row ) {
			if (array_key_exists ( "path_id", $row )) {
				$ref = new ResourceRef ( $row ["id"] );
				$ref->img = new Resource ( $row ["img_id"] );
				$ref->imgOver = new Resource ( $row ["img_over_id"] );
				$ref->imgOver->path = new Path ( $row ["path_id"] );
				$ref->imgOver->path->path = $row ["path"];
				array_push ( $refs, $ref );
			}
		}
		return $refs;
	}
	public function getResourceRefs($offset = 0, $limit = 500) {
		$query = 'SELECT ref.*,map.path FROM CONTENT_RESOURCE_REF_V2  ref left join CONTENT_RESOURCE_MAP_V2 map on ref.path_id=map.id order by comment,label LIMIT :offset,:limit';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":offset", $offset );
		$stmt->bindValue ( ":limit", $limit );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$refs = array ();
		foreach ( $rows as $row ) {
			$ref = new ResourceRef ( $row ["id"] );
			$ref->comment = $row ["comment"];
			$ref->description = $row ["description"];
			$ref->label = $row ["label"];
			$ref->title = $row ["title"];
			if ($row ["img_id"] != null) {
				$ref->img = new Resource ( $row ["img_id"] );
			}
			if ($row ["img_over_id"] != null) {
				$ref->imgOver = new Resource ( $row ["img_over_id"] );
			}
			$ref->imgAlt = $row ["img_alt"];
			if ($row ["path_id"] != null) {
				$ref->url = $row ["path"];
				$ref->path = new Path ( $row ["path_id"] );
			} else {
				$ref->url = $row ["url"];
			}
			array_push ( $refs, $ref );
		}
		return $refs;
	}
	public function createWebPage($locale, $name, $path, $template) {
		try {
			$webPage = new WebPage ( self::uuid () );
			$webPage->version = time ();
			$webPage->name = $name;
			$webPage->locale = $locale;
			$webPage->templateName = $template;
			$p = new Path ( self::uuid () );
			$p->path = $path;
			$p->resource = $webPage;
			$webPage->path = $p;
			$webPage->model = new DataList ( $this, self::uuid () );
			$webPage->data = new ResourceData ( self::uuid (), "text/html" );
			// Preview
			$webPage->preview->id = self::uuid ();
			$this->createDataPreviewDirectories ( $webPage->preview->id );
			file_put_contents ( $this->getDataFilePath ( $webPage->preview->id ), $this->config->getWebPagePreviewData () );
			$this->updateDataContentType ( $webPage->preview );
			
			$db = $this->config->db ();
			$db->beginTransaction ();
			$this->createResourceRow ( $db, $webPage );
			$this->createResourceVersionRow ( $db, $webPage );
			$this->createWebPageRow ( $db, $webPage );
			$this->createResourceMapRow ( $db, $webPage->path );
			$db->commit ();
			return $webPage;
		} catch ( Exception $e ) {
			$db->rollBack ();
			return null;
		}
		
		return null;
	}
	private function updateResourceDataFromResource(Resource $resource, $content, $contentType = null) {
		$resource->data = new ResourceData ();
		$resource->data->filename = $this->buildDefaultFilename ( $resource );
		$resource->preview = new ResourceData ();
		$resource->preview->filename = 'preview-' . $this->buildDefaultFilename ( $resource );
		$this->updateData ( $resource->data, $content, $contentType );
		$this->updateDataPreviewFromData ( $resource->preview, $resource->data );
		return $resource->data;
	}
	public function updateResourceData($resourceId, $content, $contentType = null) {
		$version = time ();
		$resource = $this->getResource ( $resourceId );
		if ($resource !== null) {
			$resource->version = time ();
			$this->updateResourceDataFromResource ( $resource, $content, $contentType );
			if ($resource->data->id !== null) {
				$this->updateDataPreviewFromData ( $resource->preview, $resource->data );
				try {
					$db = $this->config->db ();
					$db->beginTransaction ();
					$this->createResourceVersionRow ( $db, $resource );
					$this->createDataRow ( $db, $resource->data );
					if ($resource->preview !== null) {
						$this->createDataRow ( $db, $resource->preview );
					}
					$db->commit ();
					return $resource;
				} catch ( Exception $e ) {
					$db->rollBack ();
				}
			}
		}
	}
	public function updateResource(Resource $resource) {
		$resource->version = time ();
		$this->createResourceVersionRow ( $this->config->db (), $resource );
	}
	public function updateWebPageSEO(WebPage $webPage) {
		$query = 'UPDATE CONTENT_WEBPAGE_V2 set seo_title=:seoTitle,seo_description=:seoDescription,seo_img_id=:seoImageId where id=:id';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $webPage->id );
		$stmt->bindValue ( ":seoTitle", $webPage->seoTitle );
		$stmt->bindValue ( ":seoDescription", $webPage->seoDescription );
		$stmt->bindValue ( ":seoImageId", $webPage->seoImage == null ? null : $webPage->seoImage->id );
		$stmt->execute ();
	}
	public function updateWebPage(WebPage $webPage) {
		$query = 'UPDATE CONTENT_WEBPAGE_V2 set template=:template,parameters=:parameters where id=:id';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $webPage->id );
		$stmt->bindValue ( ":parameters", $webPage->parameters );
		$stmt->bindValue ( ":template", $webPage->templateName );
		$stmt->execute ();
	}
	private function createDataPreviewDirectories($dataId) {
		$dataPreviewFileName = $this->getDataFilePath ( $dataId );
		$old_umask = umask ( 0 );
		mkdir ( dirname ( $dataPreviewFileName ), 0750, true );
		umask ( $old_umask );
	}
	private function updateDataPreviewFromFile(ResourceData $preview, $file) {
		if (file_exists ( $file )) {
			$id = self::uuid ();
			$dataPreviewFileName = $this->getDataFilePath ( $id );
			$this->createDataPreviewDirectories ( $id );
			$command = null;
			$mimeReader = new MimeReader ( $file );
			switch ($mimeReader->getType ()) {
				case "application/pdf" :
					$command = "convert -quality 100 -density 300 ${file}[0] -scale 96x96 png:$dataPreviewFileName";
					break;
				default :
					$command = "convert -quality 100 -density 300 ${file} -scale 96x96 png:$dataPreviewFileName";
			}
			if ($command !== null) {
				exec ( $command );
			}
			if (file_exists ( $dataPreviewFileName )) {
				$preview->id = $id;
				$this->updateDataInfo ( $preview, $dataPreviewFileName, "image/png" );
			}
		}
		return $preview;
	}
	private function updateDataInfo(ResourceData $data, $filename, $contentType = null) {
		if ($contentType == null) {
			$this->updateDataContentType ( $data );
		} else {
			$data->contentType = $contentType;
		}
		
		if ($data->contentType != null && self::textStartsWith ( $data->contentType, "text" )) {
			// FIXME
			$data->charset = "UTF-8";
		}
		$data->date = filectime ( $filename );
		$data->contentEncoding = "identity";
		$data->length = filesize ( $filename );
		$data->md5 = md5_file ( $filename, true );
	}
	private function updateDataContentType($data) {
		$dataFileName = $this->getDataFilePath ( $data->id );
		$mimeReader = new MimeReader ( $dataFileName );
		$data->contentType = $mimeReader->getType ();
	}
	private function updateDataPreviewFromData(ResourceData $preview, ResourceData $data) {
		if ($data->id == null) {
			return;
		}
		$file = $this->getDataFilePath ( $data->id );
		$this->updateDataPreviewFromFile ( $preview, $file );
	}
	private static function textStartsWith($haystack, $needle) {
		return $needle === "" || strrpos ( $haystack, $needle, - strlen ( $haystack ) ) !== false;
	}
	private function updateData(ResourceData $data, $content, $contentType = null) {
		$id = self::uuid ();
		$dataFileName = $this->getDataFilePath ( $id );
		$old_umask = umask ( 0 );
		mkdir ( dirname ( $dataFileName ), 0750, true );
		umask ( $old_umask );
		file_put_contents ( $dataFileName, $content );
		if (file_exists ( $dataFileName )) {
			$data->id = $id;
			$this->updateDataInfo ( $data, $dataFileName, $contentType );
		}
	}
	private function buildDefaultFilename($resource) {
		$filename = strtolower ( trim ( $resource->name ) );
		$filename = str_replace ( array (
				'à',
				'á',
				'â',
				'ã',
				'å' 
		), 'a', $filename );
		$filename = str_replace ( array (
				'è',
				'é',
				'ê',
				'ë' 
		), 'e', $filename );
		$filename = str_replace ( array (
				'ì',
				'í',
				'î',
				'ï' 
		), 'i', $filename );
		$filename = str_replace ( array (
				'ò',
				'ó',
				'ô',
				'õ',
				'ø' 
		), 'o', $filename );
		$filename = str_replace ( array (
				'ù',
				'ú',
				'û' 
		), 'u', $filename );
		$filename = str_replace ( array (
				'æ',
				'œ',
				'ý',
				'ÿ',
				'ç',
				'ß',
				'ñ' 
		), array (
				'ae',
				'oe',
				'y',
				'y',
				'c',
				'ss',
				'n' 
		), $filename );
		$filename = str_replace ( array (
				'ä',
				'ö',
				'ü' 
		), array (
				'a',
				'o',
				'u' 
		), $filename );
		$filename = preg_replace ( '/[^0-9a-z-\.]/', ' ', $filename );
		$filename = trim ( $filename );
		$filename = preg_replace ( '/[ -]+/', '-', trim ( $filename ) );
		return $filename;
	}
	private function buildDefaultPath($resource) {
		$prefix = "/data/";
		$filename = $this->buildDefaultFilename ( $resource );
		return "$prefix$filename";
	}
	public function createResource($locale, $name, $content, $contentType = null) {
		$resource = new Resource ( self::uuid () );
		$resource->version = time ();
		$resource->locale = $locale;
		$resource->name = $name == null ? bin2hex ( $resource->id ) : $name;
		$this->updateResourceDataFromResource ( $resource, $content, $contentType );
		$path = new Path ( self::uuid () );
		$path->path = $this->buildDefaultPath ( $resource );
		$path->online=true;
		$path->resource = $resource;
		$resource->path = $path;
		if ($resource->data->id !== null) {
			try {
				$db = $this->config->db ();
				$db->beginTransaction ();
				$this->createResourceRow ( $db, $resource );
				$this->createResourceVersionRow ( $db, $resource );
				$this->createDataRow ( $db, $resource->data );
				if ($resource->preview !== null) {
					$this->createDataRow ( $db, $resource->preview );
				}
				$this->createResourceMapRow ( $db, $resource->path );
				$db->commit ();
				return $resource;
			} catch ( Exception $e ) {
				$db->rollBack ();
			}
		}
		return null;
	}
	public function createResourceRefFromURL($url) {
		$ref = new ResourceRef ( self::uuid () );
		$ref->url = $url;
		$db = $this->config->db ();
		$this->createResourceRefRow ( $db, $ref );
		return $ref;
	}
	public function createResourceRefFromResourceId($resourceId) {
		$resource = $this->getCurrentResource ( $resourceId );
		if ($resource != null) {
			$ref = new ResourceRef ( self::uuid () );
			$ref->path = $resource->path;
			$db = $this->config->db ();
			$this->createResourceRefRow ( $db, $ref );
			return $ref;
		}
		return null;
	}
	private function createListItemRow($db, $listId, DatalistItem $item) {
		$this->insertIndex ( $db, $listId, $item->idx );
		$query = 'INSERT INTO CONTENT_LIST_ITEM_V2(id,list_id,idx,`key`,value_resource_id,value_resource_ref_id,value_list_id,deleted) values(:id,:listId,:idx,:key,:resourceId,:resourceRefId,:subListId,:deleted)';
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $item->id );
		$stmt->bindValue ( ":listId", $listId );
		$stmt->bindValue ( ":idx", $item->idx );
		$stmt->bindValue ( ":key", $item->key );
		$stmt->bindValue ( ":deleted", $item->deleted?1:0 );
		$stmt->bindValue ( ":resourceId", ($item->data instanceof Resource) ? $item->data->id : null );
		$stmt->bindValue ( ":resourceRefId", ($item->data instanceof ResourceRef) ? $item->data->id : null );
		$stmt->bindValue ( ":subListId", ($item->data instanceof DataList) ? $item->data->id : null );
		$stmt->execute ();
	}
	private function removeIndex($db, $listId, $idx) {
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set idx=idx-1 WHERE list_id=:listId and idx>=:idx";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":listId", $listId );
		$stmt->bindValue ( ":idx", $idx );
		$stmt->execute ();
	}
	private function insertIndex($db, $listId, $idx) {
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set idx=idx+1 WHERE list_id=:listId and idx>=:idx";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":listId", $listId );
		$stmt->bindValue ( ":idx", $idx );
		$stmt->execute ();
	}
	public function addResourceItem($listId, $idx, $key, $itemId) {
		$item = new DatalistItem ( self::uuid () );
		$item->idx = $idx;
		$item->key = $key;
		$item->data = new Resource ( $itemId );
		$db = $this->config->db ();
		$this->createListItemRow ( $db, $listId, $item );
		return $item;
	}
	public function addTextItem($listId, $idx, $key, $text) {
		$item = null;
		$data = $this->createResource ( "FR", null, $text, "text/html" );
		if ($data != null) {
			$item = new DatalistItem ( self::uuid () );
			$item->idx = $idx;
			$item->key = $key;
			$item->data = $data;
			$db = $this->config->db ();
			$this->createListItemRow ( $db, $listId, $item );
		}
		return $item;
	}
	public function addResourceRefItem($listId, $idx, $key, $itemId) {
		$item = new DatalistItem ( self::uuid () );
		$item->idx = $idx;
		$item->key = $key;
		$item->data = new ResourceRef ( $itemId );
		$db = $this->config->db ();
		$this->createListItemRow ( $db, $listId, $item );
		return $item;
	}
	public function addListItem($listId, $idx, $key, $itemId) {
		$item = new DatalistItem ( self::uuid () );
		$item->idx = $idx;
		$item->key = $key;
		$item->data = new DataList ( $this, $itemId );
		$db = $this->config->db ();
		$this->createListItemRow ( $db, $listId, $item );
		return $item;
	}
	public function removeListItem(DatalistItem $item) {
		$db = $this->config->db ();
		if ($item->deleted) {
			$this->removeIndex ( $db, $item->list->id, $item->idx );
			$query = "DELETE FROM CONTENT_LIST_ITEM_V2 WHERE id=:id";
			$stmt = $db->prepare ( $query );
			$stmt->bindValue ( ":id", $id );
			$stmt->execute ();
		}
	}
	public function trashItem(DatalistItem $item) {
		$db = $this->config->db ();
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set deleted=1 WHERE id=:id";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $item->id );
		$stmt->execute ();
	}
	public function untrashItem(DatalistItem $item) {
		$db = $this->config->db ();
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set deleted=0 WHERE id=:id";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $item->id );
		$stmt->execute ();
	}
	public function updateListItem($id, $idx, $key) {
		$db = $this->config->db ();
		$item = $this->getListItem ( $id );
		$this->removeIndex ( $db, $item->list->id, $item->idx );
		$this->insertIndex ( $db, $item->list->id, $idx );
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set idx=:idx,`key`=:key  WHERE id=:id";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $id );
		$stmt->bindValue ( ":idx", $idx );
		$stmt->bindValue ( ":key", $key );
		$stmt->execute ();
	}
	public function updateListItemIdx($id, $idx) {
		$db = $this->config->db ();
		$item = $this->getListItem ( $id );
		$this->removeIndex ( $db, $item->list->id, $item->idx );
		$this->insertIndex ( $db, $item->list->id, $idx );
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set idx=:idx WHERE id=:id";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $id );
		$stmt->bindValue ( ":idx", $idx );
		$stmt->execute ();
	}
	public function updateListItemKey($id, $idx) {
		$db = $this->config->db ();
		$query = "UPDATE CONTENT_LIST_ITEM_V2 set `key`=:key WHERE id=:id";
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $id );
		$stmt->bindValue ( ":key", $key );
		$stmt->execute ();
	}
	public function getListSize($listId, $withDeleted = true) {
		$query = 'SELECT count(*) as c FROM CONTENT_LIST_ITEM_V2 WHERE list_id=:listId';
		if (! $withDeleted) {
			$query .= ' and (deleted is null or deleted = 0)';
		}
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":listId", $listId );
		$stmt->execute ();
		$row = $stmt->fetch ();
		return intval ( $row ["c"] );
	}
	public function getListItemFromKey($listId, $key, $withDeleted = true) {
		$query = 'SELECT * FROM CONTENT_LIST_ITEM_V2  WHERE list_id=:listId and `key`=:key';
		if (! $withDeleted) {
			$query .= ' and (deleted is null or deleted = 0)';
		}
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":listId", $listId );
		$stmt->bindValue ( ":key", $key );
		$stmt->execute ();
		$row = $stmt->fetch ();
		$item = null;
		if ($row) {
			$item = new DatalistItem ( $row ["id"] );
			$item->idx = intval ( $row ["idx"] );
			$item->key = $row ["key"];
			$item->deleted = $row ["deleted"];
			$item->list = new DataList ( $this, $listId );
			// resource
			if (! empty ( $row ["value_resource_id"] )) {
				$item->data = $this->getResource ( $row ["value_resource_id"] );
			}
			// resource ref
			if (! empty ( $row ["value_resource_ref_id"] )) {
				$item->data = $this->getResourceRef ( $row ["value_resource_ref_id"] );
			}
			// list
			if (! empty ( $row ["value_list_id"] )) {
				$item->data = new DataList ( $this, $row ["value_list_id"] );
			}
		}
		return $item;
	}
	public function getListItems($listId, $load = false, $offset = 0, $limit = 500, $withDeleted = true) {
		$items = array ();
		$query = 'SELECT * FROM CONTENT_LIST_ITEM_V2 
					WHERE list_id=:listId';
		if (! $withDeleted) {
			$query .= ' and (deleted is null or deleted = 0)';
		}
		$query .= ' ORDER BY idx LIMIT :offset,:limit';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":listId", $listId );
		$stmt->bindValue ( ":offset", $offset );
		$stmt->bindValue ( ":limit", $limit );
		$stmt->execute ();
		$rows = $stmt->fetchAll ();
		$list = new DataList ( $this, $listId );
		if ($rows !== null) {
			foreach ( $rows as $row ) {
				$item = new DatalistItem ( $row ["id"] );
				$item->idx = intval ( $row ["idx"] );
				$item->key = $row ["key"];
				$item->deleted = $row ["deleted"];
				$item->list = $list;
				// resource
				if (! empty ( $row ["value_resource_id"] )) {
					if ($load) {
						$item->data = $this->getResource ( $row ["value_resource_id"] );
					} else {
						$item->data = new Resource ( $row ["value_resource_id"] );
					}
					
					array_push ( $items, $item );
				}
				// resource ref
				if (! empty ( $row ["value_resource_ref_id"] )) {
					if ($load) {
						$item->data = $this->getResourceRef ( $row ["value_resource_ref_id"] );
					} else {
						$item->data = new ResourceRef ( $row ["value_resource_ref_id"] );
					}
					
					array_push ( $items, $item );
				}
				// list
				if (! empty ( $row ["value_list_id"] )) {
					$item->data = new DataList ( $this, $row ["value_list_id"] );
					array_push ( $items, $item );
				}
			}
		}
		
		return $items;
	}
	public function loadData(Resource $resource) {
		$resource->data = $this->getData ( $resource->data->id );
	}
	public function getData($id) {
		$query = 'SELECT *  FROM CONTENT_DATA_V2 WHERE id=:id';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $id );
		$stmt->execute ();
		$row = $stmt->fetch ();
		
		if ($row !== null) {
			$data = new ResourceData ( $row ["id"] );
			$data->charset = $row ["charset"];
			$data->contentType = $row ["content_type"];
			$data->date = $row ["date"];
			$data->encoding = $row ["encoding"];
			$data->filename = $row ["filename"];
			$data->length = $row ["length"];
			$data->md5 = $row ["md5"];
			return $data;
		}
		return null;
	}
	public function getListItem($itemId) {
		$query = 'SELECT * FROM CONTENT_LIST_ITEM_V2 WHERE id=:itemId';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":itemId", $itemId );
		$stmt->execute ();
		$row = $stmt->fetch ();
		if ($row !== null) {
			$item = new DatalistItem ( $row ["id"] );
			$item->idx = intval ( $row ["idx"] );
			$item->key = $row ["key"];
			$item->deleted = $row ["deleted"];
			$item->list = new DataList ( $this, $row ["list_id"] );
			// resource
			if (! empty ( $row ["value_resource_id"] )) {
				$item->data = new Resource ( $row ["value_resource_id"] );
			}
			// resource ref
			if (! empty ( $row ["value_resource_ref_id"] )) {
				$item->data = new ResourceRef ( $row ["value_resource_ref_id"] );
			}
			// list
			if (! empty ( $row ["value_list_id"] )) {
				$item->data = new DataList ( $this, $row ["value_list_id"] );
			}
			return $item;
		}
		return null;
	}
	public function getResourceRef($refId) {
		$query = 'SELECT * FROM CONTENT_RESOURCE_REF_V2 ref left join CONTENT_RESOURCE_MAP_V2 map on ref.path_id=map.id WHERE ref.id=:id';
		$db = $this->config->db ();
		$stmt = $db->prepare ( $query );
		$stmt->bindValue ( ":id", $refId );
		$stmt->execute ();
		$row = $stmt->fetch ();
		if ($row !== null) {
			$ref = new ResourceRef ( $refId );
			$ref->comment = $row ["comment"];
			$ref->description = $row ["description"];
			$ref->img = $row ["img_id"] == null ? null : $this->getCurrentResource ( $row ["img_id"] );
			$ref->imgAlt = $row ["img_alt"];
			$ref->imgOver = $row ["img_over_id"] == null ? null : $this->getCurrentResource ( $row ["img_over_id"] );
			$ref->label = $row ["label"];
			$ref->path = null;
			$ref->url = null;
			if ($row ["path_id"] !== null) {
				$ref->path = new Path ( $row ["path_id"] );
				$ref->path->path = $row ["path"];
				$ref->url = $ref->path->path;
			} else {
				$ref->url = $row ["url"];
			}
			$ref->title = $row ["title"];
			
			return $ref;
		}
		return null;
	}
	public function updateResourceRef(ResourceRef $resourceRef) {
		$this->updateResourceRefRow ( $this->config->db (), $resourceRef );
	}
}
?>